<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Post extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var string[]
     */
    protected $fillable = [
        'user_id',
        'shared_from',
        'text'
    ];

    /**
     * Booted method of the model
     *
     * @return void
     */
    protected static function booted()
    {
        static::creating(function ($post) {
            $post->user_id = auth()->user()->id;
        });

        static::deleting(function ($post) {
            foreach ($post->media as $medium) {
                $file = config('filesystems.disks.' . env('UPLOADS_DISK') . '.root') . $medium;

                if (file_exists($file)) {
                    unlink($file);
                }

                $medium->delete();
            }
        });
    }

    /**
     * Get the user that owns the post.
     *
     * @return BelongsTo
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the media for the post.
     *
     * @return HasMany
     */
    public function media(): HasMany
    {
        return $this->hasMany(PostMedia::class);
    }

    /**
     * Get the shared post
     *
     * @return BelongsTo
     */
    public function sharedPost(): BelongsTo
    {
        return $this->belongsTo(Post::class, 'shared_from');
    }

    /**
     * Retrieve associated comments
     *
     * @return MorphMany
     */
    public function comments(): MorphMany
    {
        return $this->morphMany(Comment::class, 'commentable');
    }

    /**
     * Retrieve associated reacts
     *
     * @return MorphMany
     */
    public function reacts(): MorphMany
    {
        return $this->morphMany(React::class, 'reactable');
    }
}
