<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;

class Court extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'club_id',
        'user_id',
        'name',
        'location',
        'latitude',
        'longitude',
        'description',
        'amenities',
        'surfaces',
        'nets',
        'no_indoor_courts',
        'no_outdoor_courts',
        'city',
        'state',
        'zip',
        'phone',
        'email',
        'website',
        'status',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        // Amenities and nets are stored as comma-separated strings
    ];

    protected $appends = ['is_favorite'];

    /**
     * Get the club that owns the court.
     *
     * @return BelongsTo
     */
    public function club(): BelongsTo
    {
        return $this->belongsTo(Club::class);
    }

    /**
     * Get the media for the court.
     *
     * @return HasMany
     */
    public function media(): HasMany
    {
        return $this->hasMany(CourtMedia::class);
    }
    /**
     * Retrieve associated reviews
     *
     * @return MorphMany
     */
    public function reviews(): MorphMany
    {
        return $this->morphMany(Review::class, 'rateable');
    }

    public function getIsFavoriteAttribute()
    {
        $fav = Favorite::where([
            'user_id' => auth()->user()->id,
            'favoritable_type' => 'App\Models\Court',
            'favoritable_id' => $this->id
        ])
            ->first();
        if ($fav) {
            return true;
        }
        return false;
    }

    /**
     * Get amenities as an array from comma-separated string
     *
     * @return array
     */
    public function getAmenitiesArrayAttribute()
    {
        return $this->amenities ? explode(',', $this->amenities) : [];
    }

    /**
     * Get nets as an array from comma-separated string
     *
     * @return array
     */
    public function getNetsArrayAttribute()
    {
        return $this->nets ? explode(',', $this->nets) : [];
    }

    /**
     * Set amenities from array to comma-separated string
     *
     * @param array $amenities
     */
    public function setAmenitiesFromArray($amenities)
    {
        $this->amenities = is_array($amenities) ? implode(',', $amenities) : $amenities;
    }

    /**
     * Set nets from array to comma-separated string
     *
     * @param array $nets
     */
    public function setNetsFromArray($nets)
    {
        $this->nets = is_array($nets) ? implode(',', $nets) : $nets;
    }
}
