<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class Club extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'name',
        'logo',
        'location',
        'latitude',
        'longitude',
        'description',
        'total_courts',
        'amenities',
        'surfaces',
        'nets',
        'status',
        'city',
        'state',
        'zip',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'total_courts' => 'integer',
    ];

    protected $appends = [
        'is_favorite'
    ];


    /**
     * Booted method of the model
     */
    // protected static function booted(): void
    // {
    //     static::creating(function (Club $model) {
    //         $model->user_id = auth()->user()->id;
    //     });
    // }

    /**
     * Get the club logo
     *
     * @return Attribute
     */

    protected function logo(): Attribute
    {
        return new Attribute(
            get: fn ($value) => $value != '' ? env('UPLOAD_URL') . $value : env('UPLOAD_URL').'clubs/default-logo.jpg',
        );
    }

    /**
     * Get the user that owns the club.
     * 
     * @return BelongsTo
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function getIsFavoriteAttribute()
    {
        $fav = Favorite::where([
            'user_id' => auth()->user()->id, 
            'favoritable_type' => 'App\Models\Club', 
            'favoritable_id' => $this->id])
            ->first();
        if($fav)
        {
            return true;
        }
            return false;
    }


    /**
     * Get the working hours for the club.
     * 
     * @return HasMany
     */
    public function workingHours(): HasMany
    {
        return $this->hasMany(WorkingHour::class);
    }

    /**
     * Get the features for the club.
     * 
     * @return BelongsToMany
     */
    public function features(): BelongsToMany
    {
        return $this->belongsToMany(Feature::class)->using(ClubFeature::class);
    }

    /**
     * Get the courts for the club.
     * 
     * @return HasMany
     */
    public function courts(): HasMany
    {
        return $this->hasMany(Court::class);
    }

    /**
     * Retrieve associated favorites
     *
     * @return MorphMany
     */
    public function favorites(): MorphMany
    {
        return $this->morphMany(Favorite::class, 'favoritable');
    }

    /**
     * Retrieve associated reservations
     *
     * @return HasMany
     */
    public function reservations(): HasMany
    {
        return $this->hasMany(Reservation::class);
    }

    /**
     * Retrieve associated reviews
     *
     * @return MorphMany
     */
    public function reviews(): MorphMany
    {
        return $this->morphMany(Review::class, 'rateable');
    }

    /**
     * Retrieve associated tournaments
     *
     * @return HasMany
     */
    public function tournaments(): HasMany
    {
        return $this->hasMany(Tournament::class);
    }

    public function events(): HasMany
    {
        return $this->hasMany(Event::class);
    }
}
