<?php

namespace App\Models;

use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Ad extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var string[]
     */
    protected $fillable = [
        'user_id',
        'card_id',
        'title',
        'slug',
        'description',
        'days',
        'amount',
        'currency',
        'status', // 0 = pending, 1 = active, 2 = inactive, 3 = expired, 4 = cancelled
        'redirect_url',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'is_active' => 'boolean',
    ];

    /**
     * The "booted" method of the model.
     *
     * @return void
     */
    protected static function booted()
    {
        static::creating(function ($ad) {
            $ad->user_id = auth()->user()->id;

            if (self::where('slug', Str::slug($ad->title))->exists()) {
                $ad->slug = Str::slug($ad->title) . '-' . random_int(1, 9999);
            } else {
                $ad->slug = Str::slug($ad->title);
            }
        });

        static::updating(function ($ad) {
            if (self::where('slug', Str::slug($ad->title))->exists()) {
                $ad->slug = Str::slug($ad->title) . '-' . random_int(1, 9999);
            } else {
                $ad->slug = Str::slug($ad->title);
            }
        });

        static::deleting(function ($ad) {
            foreach ($ad->adMedia as $medium) {
                $file = config('filesystems.disks.' . env('UPLOADS_DISK') . '.root') . $medium;

                if (file_exists($file)) {
                    unlink($file);
                }

                $medium->delete();
            }
        });
    }

    /**
     * Get the user that owns the ad.
     *
     * @return BelongsTo
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the associated ad media.
     *
     * @return HasMany
     */
    public function adMedia(): HasMany
    {
        return $this->hasMany(AdMedia::class);
    }
}
