<?php

namespace App\Http\Controllers;

use App\Models\Club;
use App\Models\Reservation;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Yajra\DataTables\Facades\DataTables;

class ReservationController extends Controller
{
    protected $module, $data = [];

    public function __construct()
    {
        $this->module = new Reservation();
        $this->data['module_name'] = 'Reservation';
        $this->data['view_directory'] = 'pages.reservation';
        $this->data['parent_named_route'] = 'reservations';
    }

    public function index(Request $request)
    {
        if ($request->ajax()) {

            if (auth()->user()->hasRole('club')) {
                $user_clubs = auth()->user()->clubs->pluck('id');
                $data = $this->module::with('user', 'club')->whereIn('club_id', $user_clubs);
            } else {
                $data = $this->module::with('user', 'club');
            }

            return DataTables::of($data)
                ->addIndexColumn()
                ->addColumn('sno', function ($row) {
                    static $index = 0; // Static variable to keep track of the index
                    $page = request()->get('start', 0); // Get the start parameter
                    $index++;
                    return $page + $index;
                })
                ->addColumn('action', function ($row) {
                    $edit = '<a href="' . route($this->data['parent_named_route'] . ".edit", ['id' => $row->id]) . '" class="edit btn btn-primary btn-sm btn-equal mb-1">Edit</a>&nbsp;';

                    $delete = '<a href="#" onclick="confirmDelete(\'' . $this->data['parent_named_route'] . '\', ' . $row->id . ')" class="edit btn btn-danger btn-sm btn-equal mb-1">Delete</a>';
                    return $delete . $edit;
                })
                ->addColumn('user_name', function ($row) {
                    return $row->user->name;
                })
                ->addColumn('day', function ($row) {
                    return $this->module::DAYS[$row->day];
                })
                ->addColumn('club_name', function ($row) {
                    return $row->club->name;
                })
                ->addColumn('reservation_date', function ($row) {
                    return customDateFormat($row->reservation_date);
                })
                ->addColumn('start_time', function ($row) {
                    return customTimeFormat($row->start_time);
                })
                ->addColumn('end_time', function ($row) {
                    return customTimeFormat($row->end_time);
                })
                ->rawColumns(['action', 'user_name', 'club_name', 'reservation_date', 'end_time', 'start_time'])
                ->make(true);
        }

        $this->data['page'] =  'List';

        return view($this->data['view_directory'] . '.' . __FUNCTION__, $this->data);
    }

    public function create(Request $request)
    {
        if ($request->isMethod('post')) {
            $validatedData = $request->validate([
                'club_id'       => 'required',
                'day'       => 'required|string|max:255',
                'reservation_date' => 'required|date_format:Y-m-d', // Adjust format as needed
                'start_time' => 'required|date_format:H:i', // Adjust format as needed
                'end_time'   => 'required|date_format:H:i', // Adjust format as needed
            ]);

            $validatedData['start_time'] = Carbon::createFromFormat('H:i', $validatedData['start_time'])->format('H:i:s');
            $validatedData['end_time'] = Carbon::createFromFormat('H:i', $validatedData['end_time'])->format('H:i:s');
            $validatedData['user_id'] = Auth::id();
            $validatedData['status'] = 'accepted';

            $reservation = $this->module->create($validatedData);

            return redirect()->route('reservations.index')->with('success', $this->data['module_name'] . ' added successfully');
        }

        $this->data['users'] = User::all();
        if (auth()->user()->hasRole('club')) {
            $this->data['clubs'] = auth()->user()->clubs;
        } else {
            $this->data['clubs'] = Club::all();
        }
        return view($this->data['view_directory'] . '.' . __FUNCTION__, $this->data);
    }

    public function edit(Request $request, $id)
    {
        if ($request->isMethod('post')) {

            $validatedData = $request->validate([
                'club_id'       => 'required',
                'day'       => 'required|string|max:255',
                'reservation_date' => 'required|date_format:Y-m-d', // Adjust format as needed
                'start_time' => 'required|date_format:H:i', // Adjust format as needed
                'end_time'   => 'required|date_format:H:i', // Adjust format as needed
                'status'   => 'required|string',
            ]);


            // Convert time to H:i:s format
            $validatedData['start_time'] = Carbon::createFromFormat('H:i', $validatedData['start_time'])->format('H:i:s');
            $validatedData['end_time'] = Carbon::createFromFormat('H:i', $validatedData['end_time'])->format('H:i:s');
            $validatedData['user_id'] = Auth::id();

            $tournament = $this->module->where('id', $id)->update($validatedData);

            return redirect()->route('reservations.index')->with('success', $this->data['module_name'] . ' updated successfully');
        }

        $reservation = $this->module::find($id);
        $this->data['reservation'] = $this->module::find($id);

        if (auth()->user()->hasRole('club')) {
            $this->data['clubs'] = auth()->user()->clubs;
            if (!$this->data['reservation'] || $this->data['reservation']->club->user->id !== auth()->user()->id) {
                abort(403, 'You do not have permission to ' . __FUNCTION__ . ' this ' . $this->data['module_name']);
            }
        } else {
            $this->data['clubs'] = Club::all();
        }

        $this->data['days'] = $this->module::DAYS;
        $this->data['statuses'] = $this->module::STATUS;
        return view($this->data['view_directory'] . '.' . __FUNCTION__, $this->data);
    }

    public function view(Request $request)
    {
        $this->data['comment'] = $this->module::with('parent', 'user')->find($request->id);

        $this->data['page'] = __FUNCTION__;

        return view($this->data['view_directory'] . '.' . __FUNCTION__, $this->data);
    }

    public function status($id)
    {
        $record = $this->module::find($id);
        $record->status = $record->status == 'accepted' ? 'cancelled' : 'accepted';
        $record->save();

        return redirect()->back()->with('success', $this->data['module_name'] . ' status updated successfully');
    }

    public function delete($id)
    {
        $record = $this->module::find($id);

        if ($record) {
            $record->delete();

            return redirect()->back()->with('success', $this->data['module_name'] . ' deleted successfully');
        }

        return redirect()->back()->with('error', $this->data['module_name'] . ' not found');
    }
}
