<?php

namespace App\Http\Controllers;

use App\Models\Group;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class GroupController extends Controller
{
    protected $module, $data = [];

    public function __construct()
    {
        $this->module = new Group();
        $this->data['module_name'] = 'Group';
        $this->data['view_directory'] = 'pages.group';
        $this->data['parent_named_route'] = 'groups';
    }

    public function index(Request $request)
    {
        if ($request->ajax()) {

            $data = $this->module::with('user')
                ->select([
                    'groups.*',
                    DB::raw("users.name AS full_name")
                ])
                ->leftJoin('users', 'groups.user_id', '=', 'users.id');

            // $sno = 1;
            return DataTables::of($data)
                ->addIndexColumn()
                ->addColumn('sno', function ($row) {
                    static $index = 0; // Static variable to keep track of the index
                    $page = request()->get('start', 0); // Get the start parameter
                    $index++;
                    return $page + $index;
                })
                ->addColumn('action', function ($row) {

                    $edit = '<a href="' . route("groups.edit", ['id' => $row->id]) . '" class="edit btn btn-info btn-sm btn-equal mb-1">Edit</a>&nbsp;';

                    $delete = '<a href="#" onclick="confirmDelete(\'' . $this->data['parent_named_route'] . '\', ' . $row->id . ')" class="edit btn btn-danger btn-sm btn-equal mb-1">Delete</a>';

                    return $edit . $delete;
                })
                ->addColumn('is_joined', function ($row) {
                    return $row->is_joined
                        ? '<i style="color:green;" class="menu-icon tf-icons bx bx-user-check"></i> Joined'
                        : '<i style="color:red;" class="menu-icon tf-icons bx bx-user-x"></i> Not Joined';
                })
                ->rawColumns(['action', 'is_joined'])
                ->filterColumn('full_name', function ($query, $keyword) {
                    $query->whereRaw("users.name like ?", ["%{$keyword}%"]);
                })
                ->make(true);
        }

        $this->data['page'] =  'List';

        return view($this->data['view_directory'] . '.' . __FUNCTION__, $this->data);
    }

    public function create(Request $request)
    {
        if ($request->isMethod('post')) {
            $this->validate($request, [
                'name' => 'required|string|max:255',
                'description' => 'required|string|max:255',
            ]);

            $input = $request->all();
            $input['user_id'] = Auth::user()->id;
            $this->module::create($input);

            return redirect()->route($this->data['parent_named_route'] . '.index')->with('success', $this->data['module_name'] . ' created successfully');
        }

        return view($this->data['view_directory'] . '.' . __FUNCTION__, $this->data);
    }

    public function edit(Request $request, $id)
    {
        if ($request->isMethod('post')) {

            $this->validate($request, [
                'name' => 'required|string|max:255',
                'description' => 'required|string|max:255',
            ]);

            $input = $request->except('_token');
            $this->module::where('id', $id)->update($input);

            return redirect()->route($this->data['parent_named_route'] . '.index')->with('success', $this->data['module_name'] . ' updated successfully');
        }

        $this->data['group'] = $this->module::find($id);

        return view($this->data['view_directory'] . '.' . __FUNCTION__, $this->data);
    }

    public function status($id)
    {
        $user = $this->module::find($id);
        $user->status = $user->status ? '0' : '1';
        $user->save();

        return redirect()->back()->with('success', $this->data['module_name'] . ' status updated successfully');
    }

    public function delete($id)
    {
        $user = $this->module::find($id);

        if ($user) {
            $user->delete();

            return redirect()->back()->with('success', $this->data['module_name'] . ' deleted successfully');
        }

        return redirect()->back()->with('error', 'User not found');
    }
}
