<?php

namespace App\Http\Controllers;

use App\Models\Club;
use App\Models\Event;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Yajra\DataTables\Facades\DataTables;

class EventController extends Controller
{
    protected $module, $data = [];

    public function __construct()
    {
        $this->module = new Event();
        $this->data['module_name'] = 'Event';
        $this->data['view_directory'] = 'pages.event';
        $this->data['parent_named_route'] = 'events';
    }

    public function index(Request $request)
    {
        if ($request->ajax()) {

            if (auth()->user()->hasRole('club')) {
                $user_clubs = auth()->user()->clubs->pluck('id');
                $data = $this->module::with('user', 'club')->whereIn('club_id', $user_clubs);
            } else {
                $data = $this->module::with('user', 'club');
            }

            return DataTables::of($data)
                ->addIndexColumn()
                ->addColumn('sno', function ($row) {
                    static $index = 0; // Static variable to keep track of the index
                    $page = request()->get('start', 0); // Get the start parameter
                    $index++;
                    return $page + $index;
                })
                ->addColumn('action', function ($row) {
                    $edit = '<a href="' . route($this->data['parent_named_route'] . ".edit", ['id' => $row->id]) . '" class="edit btn btn-primary btn-sm btn-equal mb-1">Edit</a>&nbsp;';
                    // $delete = '<a href="' . route($this->data['parent_named_route'] . ".delete", ['id' => $row->id]) . '" class="edit btn btn-danger btn-sm btn-equal mb-1">Delete</a>';
                    $delete = '<a href="#" onclick="confirmDelete(\'' . $this->data['parent_named_route'] . '\', ' . $row->id . ')" class="edit btn btn-danger btn-sm btn-equal mb-1">Delete</a>';
                    return $edit . $delete;
                })
                ->addColumn('user_name', function ($row) {
                    return $row->user->name;
                })
                ->addColumn('club_name', function ($row) {
                    return $row->club->name;
                })
                ->addColumn('start_date_time', function ($row) {
                    return customDateFormat($row->start_date) . ' ' . customTimeFormat($row->start_time);
                })
                ->addColumn('picture', function ($row) {
                    return '<img src="' . $row->picture . '" class="" width=40px" heigt=40px"/>';
                })
                ->addColumn('end_date_time', function ($row) {
                    return customDateFormat($row->end_date) . ' ' . customTimeFormat($row->end_time);
                })
                ->addColumn('status', function ($row) {
                    if ($row->status === 0) {
                        return '<i style="color:orange;" class="menu-icon tf-icons bx bx-time"></i> Pending';
                    } elseif ($row->status === 1) {
                        return '<i style="color:green;" class="menu-icon tf-icons bx bx-user-check"></i> Active';
                    } elseif ($row->status === 2) {
                        return '<i style="color:blue;" class="menu-icon tf-icons bx bx-flag"></i> Finished';
                    } else {
                        return '<i style="color:grey;" class="menu-icon tf-icons bx bx-question-mark"></i> Unknown';
                    }
                })

                ->rawColumns(['action', 'club_name', 'user_name', 'picture', 'status'])
                ->make(true);
        }

        $this->data['page'] =  'List';

        return view($this->data['view_directory'] . '.' . __FUNCTION__, $this->data);
    }

    public function create(Request $request)
    {
        if ($request->isMethod('post')) {
            $validatedData = $request->validate([
                'club_id'       => 'required',
                'name'       => 'required|string|max:255',
                'description'       => 'required|string|max:255',
                'picture'    => 'nullable|image|mimes:jpeg,png,jpg', // Max size 2MB
                'slug'       => 'required|string|max:255|unique:events,slug', // Ensure slug is unique in the table
                'start_date' => 'required|date_format:Y-m-d', // Adjust format as needed
                'end_date'   => 'required|date_format:Y-m-d',
                'start_time' => 'required|date_format:H:i', // Adjust format as needed
                'end_time'   => 'required|date_format:H:i', // Adjust format as needed
            ]);

            // Convert time to H:i:s format
            $validatedData['start_time'] = Carbon::createFromFormat('H:i', $validatedData['start_time'])->format('H:i:s');
            $validatedData['end_time'] = Carbon::createFromFormat('H:i', $validatedData['end_time'])->format('H:i:s');

            if ($request->hasFile('picture')) {
                $validatedData['picture'] = $this->media($request, 'events');
            }

            $validatedData['user_id'] = Auth::id();

            $this->module->create($validatedData);

            return redirect()->route('events.index')->with('success', $this->data['module_name'] . ' added successfully');
        }

        if (auth()->user()->hasRole('club')) {
            $this->data['clubs'] = auth()->user()->clubs;
        } else {
            $this->data['clubs'] = Club::all();
        }

        return view($this->data['view_directory'] . '.' . __FUNCTION__, $this->data);
    }

    public function edit(Request $request, $id)
    {
        if ($request->isMethod('post')) {

            $event = $this->module::find($id);

            $validatedData = $request->validate([
                'club_id'       => 'required',
                'name'       => 'required|string|max:255',
                'description'       => 'required|string|max:255',
                'picture'    => 'nullable|image|mimes:jpeg,png,jpg', // Max size 2MB
                'slug' => [
                    'required',
                    'string',
                    'max:255',
                    Rule::unique('events', 'slug')->ignore($event->id)
                ],
                'start_date' => 'required|date_format:Y-m-d', // Adjust format as needed
                'end_date'   => 'required|date_format:Y-m-d',
                'start_time' => 'required|date_format:H:i', // Adjust format as needed
                'end_time'   => 'required|date_format:H:i', // Adjust format as needed
            ]);

            // Convert time to H:i:s format
            $validatedData['start_time'] = Carbon::createFromFormat('H:i', $validatedData['start_time'])->format('H:i:s');
            $validatedData['end_time'] = Carbon::createFromFormat('H:i', $validatedData['end_time'])->format('H:i:s');

            if ($request->hasFile('picture')) {
                $validatedData['picture'] = $this->media($request, 'events');
                if ($event->picture) {
                    $mediaPath = str_replace(env('UPLOAD_URL'), '', $event->picture);
                    if (Storage::disk(env('UPLOADS_DISK'))->exists($mediaPath)) {
                        Storage::disk(env('UPLOADS_DISK'))->delete($mediaPath);
                    }
                }
            }


            $this->module->where('id', $id)->update($validatedData);

            return redirect()->route('events.index')->with('success', $this->data['module_name'] . ' updated successfully');
        }

        $this->data['event'] = Event::find($id);

        if (auth()->user()->hasRole('club')) {
            $this->data['clubs'] = auth()->user()->clubs;
            if (!$this->data['event'] || $this->data['event']->club->user->id !== auth()->user()->id) {
                abort(403, 'You do not have permission to ' . __FUNCTION__ . ' this ' . $this->data['module_name']);
            }
        } else {
            $this->data['clubs'] = Club::all();
        }

        return view($this->data['view_directory'] . '.' . __FUNCTION__, $this->data);
    }


    public function delete($id)
    {
        $recor = $this->module::find($id);

        if ($recor) {
            $recor->delete();

            return redirect()->back()->with('success', $this->data['module_name'] . ' deleted successfully');
        }

        return redirect()->back()->with('error', $this->data['module_name'] . ' not found');
    }
}
