<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Club;
use App\Models\User;
use App\Models\WorkingHour;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Where to redirect users after registration.
     *
     * @var string
     */
    protected $redirectTo = '/';

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest');
    }

    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    protected function validator(array $data)
    {
        return Validator::make($data, [
            'name' => ['required', 'string', 'max:255'],
            'picture'       => 'required|image|mimes:jpeg,png,jpg',
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:8'],

            'club_name'       => 'required|string',
            'logo'       => 'required|image|mimes:jpeg,png,jpg',
            'location'       => 'required|string',
            'latitude'       => 'required|string',
            'longitude'       => 'required|string',
            'description'       => 'required|string',
            'total_courts' => 'required|integer',
            'zip' => 'nullable|string',
        ]);
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\Models\User
     */
    protected function create(array $data)
    {
        $request = request();

        if ($request->hasFile('picture')) {
            $data['picture'] = $this->media($request, 'users');
        }

        $user = User::create([
            'name' => $data['name'],
            'user_type' => 'club',
            'email' => $data['email'],
            'picture' => $data['picture'],
            'password' => Hash::make($data['password']),
        ]);

        $user->assignRole('club'); // Replace 'club_owner' with your role name

        if ($request->hasFile('logo')) {
            $data['logo'] = $this->uploadLogo($request, 'clubs');
        }
        
        $amenitiesJson = $this->extractTagifyValues($request->input('amenities'));
        $surfaceJson   = $this->extractTagifyValues($request->input('surface'));
        $netJson       = $this->extractTagifyValues($request->input('net'));

        $data['user_id'] = $user->id;

        $club = Club::create([
            'user_id'       => $data['user_id'],
            'name'       => $data['club_name'],
            'logo'       => $data['logo'],
            'location'       => $data['location'],
            'latitude'       => $data['latitude'],
            'longitude'       => $data['longitude'],
            'description'       => $data['description'],
            'total_courts' => $data['total_courts'],
            'zip' => $data['zip'],
            'state' => $data['state'],
            'city' => $data['city'],
            'amenities' => $amenitiesJson,
            'surfaces' => $surfaceJson,
            'nets' => $netJson,
        ]);
        
        // Default working hours
$openingTime = '09:00:00';
$closingTime = '17:00:00';
$now = Carbon::now();

// Create working hours for each day (0 = Sunday, 6 = Saturday)
$workingHours = [];
for ($day = 0; $day <= 6; $day++) {
    $workingHours[] = [
        'club_id'      => $club->id,
        'day'          => $day,
        'opening_time' => $openingTime,
        'closing_time' => $closingTime,
        'created_at'   => $now,
        'updated_at'   => $now,
    ];
}

// Insert all at once
WorkingHour::insert($workingHours);
        
        

        return $user;
    }
}
