# Advanced Court Management System Setup Guide

## Overview

This guide will help you set up the advanced court management system with Google Places API integration and image cropping capabilities.

## Prerequisites

-   Laravel 8+ application
-   PHP 7.4+
-   MySQL database
-   Google Maps API key with Places API enabled

## Google API Setup

### 1. Create Google Cloud Project

1. Go to [Google Cloud Console](https://console.cloud.google.com/)
2. Create a new project or select an existing one
3. Enable the following APIs:
    - Maps JavaScript API
    - Places API
    - Geocoding API

### 2. Create API Key

1. Go to **Credentials** in the Google Cloud Console
2. Click **Create Credentials** > **API Key**
3. Copy the generated API key
4. (Optional) Restrict the API key to your domain for security

### 3. Environment Configuration

Add the following to your `.env` file:

```env
GOOGLE_MAPS_API_KEY=your_google_maps_api_key_here
GOOGLE_PLACES_API_KEY=your_google_places_api_key_here
```

Note: If you use the same key for both services, you only need to set `GOOGLE_MAPS_API_KEY`.

## Database Setup

### 1. Run Migrations

Ensure you have the latest migrations for courts and court media:

```bash
php artisan migrate
```

### 2. Court Media Table

The system uses a `court_media` table to store image information. Ensure it has these columns:

-   `id`
-   `court_id` (foreign key)
-   `file_name`
-   `file_path`
-   `file_type`
-   `mime_type`
-   `file_size`
-   `is_primary` (boolean)
-   `status`
-   `created_at`
-   `updated_at`
-   `deleted_at` (for soft deletes)

## File Storage Setup

### 1. Create Storage Directory

```bash
mkdir -p storage/app/public/courts
php artisan storage:link
```

### 2. Set Permissions

```bash
chmod -R 755 storage/
chmod -R 755 bootstrap/cache/
```

## Configuration Files

### 1. Update config/services.php

The system automatically adds Google API configuration to `config/services.php`:

```php
'google' => [
    'maps_api_key' => env('GOOGLE_MAPS_API_KEY'),
    'places_api_key' => env('GOOGLE_PLACES_API_KEY', env('GOOGLE_MAPS_API_KEY')),
],
```

### 2. Update config/filesystems.php

Ensure your public disk is configured correctly:

```php
'public' => [
    'driver' => 'local',
    'root' => storage_path('app/public'),
    'url' => env('APP_URL').'/storage',
    'visibility' => 'public',
],
```

## Routes Setup

The system adds this route to your web.php:

```php
Route::any('/create-advanced', [CourtController::class, 'createCourt'])->name('create.advanced');
```

## Assets

### CSS and JavaScript Files

The system includes:

-   `public/assets/css/court-advanced.css` - Custom styling
-   `public/assets/js/court-advanced.js` - Main functionality

### External Dependencies

-   Cropper.js for image cropping
-   Google Maps JavaScript API
-   Bootstrap 5 (assumed to be already included)

## Usage

### 1. Access the Form

Navigate to: `/admin/courts/create-advanced`

To view your courts: `/admin/courts/my-courts`

### 2. Form Features

-   **User-based Courts**: Courts are automatically associated with the authenticated user
-   **Google Places Autocomplete**: Start typing an address
-   **Automatic Geolocation**: Lat/lng, city, state, zip auto-populated
-   **Image Upload**: Upload up to 10 images (5MB each)
-   **Image Cropping**: Crop images to 16:9 aspect ratio
-   **Form Validation**: Client-side and server-side validation

## Troubleshooting

### Google API Issues

1. **Places not loading**: Check API key and enabled services
2. **Quota exceeded**: Monitor usage in Google Cloud Console
3. **CORS errors**: Ensure domain is whitelisted

### Image Upload Issues

1. **File size errors**: Increase `upload_max_filesize` in php.ini
2. **Permission errors**: Check storage directory permissions
3. **Memory errors**: Increase `memory_limit` in php.ini

### Database Issues

1. **Migration errors**: Ensure proper database connection
2. **Foreign key errors**: Check club_id references

## Security Considerations

### 1. API Key Security

-   Restrict API keys to specific domains
-   Monitor API usage regularly
-   Use environment variables, never hardcode

### 2. File Upload Security

-   Validate file types server-side
-   Limit file sizes
-   Store files outside web root when possible
-   Scan uploaded files for malware

### 3. Input Validation

-   All form inputs are validated server-side
-   SQL injection protection via Eloquent ORM
-   XSS protection via Laravel's built-in escaping

## Performance Optimization

### 1. Image Processing

-   Images are automatically compressed during cropping
-   Consider implementing image optimization queue jobs for large uploads

### 2. API Calls

-   Implement caching for frequently accessed places
-   Use API call batching when possible

### 3. Database

-   Add indexes on frequently queried columns
-   Consider pagination for large datasets

## Maintenance

### Regular Tasks

1. Clean up orphaned media files
2. Monitor API usage and costs
3. Update dependencies regularly
4. Backup uploaded images

### Monitoring

-   Set up logging for API failures
-   Monitor storage usage
-   Track form submission success rates

## Support

For issues or questions:

1. Check Laravel logs: `storage/logs/laravel.log`
2. Monitor browser console for JavaScript errors
3. Verify Google API quotas and billing

## Version History

-   v1.0 - Initial implementation with Google Places and image cropping
-   v1.1 - Updated to store amenities and nets as comma-separated strings
-   Features: Advanced form validation, geolocation, media management

---

_Last updated: [Current Date]_
_Compatible with: Laravel 8+, PHP 7.4+_
